% Copyright 2016 Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000 with Sandia
% Corporation, the U.S. Government retains certain rights in this software

% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are
% met:
% 
%     (1) Redistributions of source code must retain the above copyright
%     notice, this list of conditions and the following disclaimer. 
% 
%     (2) Redistributions in binary form must reproduce the above copyright
%     notice, this list of conditions and the following disclaimer in
%     the documentation and/or other materials provided with the
%     distribution.  
%     
%     (3)The name of the author may not be used to
%     endorse or promote products derived from this software without
%     specific prior written permission.
% 
% THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
% IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
% WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
% DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
% INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
% (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
% SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
% HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
% STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
% IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
% POSSIBILITY OF SUCH DAMAGE.

function deconData = despikeKatsumoto(data,m,sigmaThreshold)

%FUNCTION deconData = despikeKatsumoto(data,m,sigmaThreshold)
%
% PURPOSE:
%	despikeKatsumoto.m is designed to remove cosmic ray spikes from
%	spectra. 
%
% DEPENDENCIES:
%	-None-
%
% CLASSES:
%	-None-
%
% INPUTS:
%	data:
%		An m by n matrix, where each of the n columns corresponds to a 
%		spectrum. The data is expected to be contaminated with cosmic
%		spikes. 
%	m:
%		A variable specifying the size of the filter span, which is set
%		at 2*m+1. 
%	sigmaThreshold:
%		A scaling factor that determines how aggressively spikes are
%		removed, an adjustable parameter of the algorithm. This parameter
%		appears to control (approximately) how many standard deviations a
%		spike has to be away from the smoothed curve to register as a
%		spike. 
%
% OUTPUTS:
%	deconData:
%		An m by n matrix, where each of the n columns corresponds to a 
%		spectrum. In the filtered output, the cosmic spikes in the
%		original data have hopefully been suppressed. 
%
% REFERENCES:
%   This code was designed as an implementation of the following reference:
%	1)	Katsumoto, Y. and Y. Ozaki (2003). "Practical algorithm for 
%		reducing convex spike noises on a spectrum." Applied Spectroscopy 
%		57(3): 317-322.
%
% PROGRAMMER COMMENTS:
%	This algorithm appears to be quite sensitive to the input parameters. 
%
% LIMITATIONS:
%	-None-

% Revision:
%	Sandia Hyper-UBS 1.0.0

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Apply the initial smoothing function. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

dataSize = size(data);
sPrime = NaN(dataSize);

%Calculate the moving-average smoothed version of the data.
for j=1:dataSize(2);
	sPrime(:,j) = smooth(data(:,j),2*m+1,'moving');
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Determine the residuals, the difference between the raw data and the
%smoothed data. Use this to compute the threshold delta. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

epsilon = data - sPrime;

%Determine its variance. 
varEpsilon = var(epsilon(:));

%Determine the threshold, delta. 
delta = sigmaThreshold*sqrt(varEpsilon);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Calculate the second smoothed term (equation 3 of the paper)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%Initially, all weights are 1 and all entries are from sPrime. 
sPrimeUsed = sPrime;
weights = ones(dataSize);

%Any entries which are above the threshold have 0 weight. 
zeroWeight = epsilon>delta;
sPrimeUsed(zeroWeight) = 0;
weights(zeroWeight) = 0;

%Take the moving average of the data and of the weights. 
sPrimeAvg = NaN(dataSize);
for j=1:dataSize(2);
	sPrimeAvg(:,j) = smooth(sPrimeUsed(:,j),2*m+1,'moving');
end
weightsAvg = NaN(dataSize);
for j=1:dataSize(2);
	weightsAvg(:,j) = smooth(weights(:,j),2*m+1,'moving');
end

%Dividing by the weight will give the desired moving average with adjusted
%weight. 
sPrimeAvg = sPrimeAvg./weightsAvg;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Calculate the estimated spectrum of the convex noise. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%Initialize
Xi = zeros(dataSize);

%Find where spikes likely are present
difference2 = data - sPrimeAvg;
spikes = difference2 >= delta;

%Where spikes are detected, set Xi equal to the difference2
Xi(spikes) = difference2(spikes);

%Determine the corrected data
deconData = data - Xi;



